unit Sticky;

interface
Uses
  Windows, Registry, Classes;

Type
  TStickyElementMode = (semHandle, semClass, semEXPR);
  TStickyElement = class(TObject)
    private
      FHandle     : THandle;
      FEXEName    : String;
      FExpression : String;
      FMode       : TStickyElementMode;
    public
      constructor Create(aHandle:THandle; Const aEXEname, aExpr: String; aMode: TStickyElementMode);
      function IsSticky(aHandle: THandle):Boolean;
      procedure Save(reg: TRegistry; No:Integer);
      constructor Load(reg: TRegistry; No:Integer);
      property Mode:TStickyElementMode read FMode write FMode;
      property Handle:THandle read FHandle write FHandle;
      property EXEName:String read FEXEName write FEXEName;
      property Expression:String read FExpression write FExpression;
  End;

  TStickyWindows = class(TObject)
    private
      FWindows : TList;
      function RItems(Index:Integer):TStickyElement;
      function RCount:Integer;
    public
      constructor Create;
      destructor Destroy; override;
      function IsSticky(aHandle: THandle):Boolean;
      procedure Remove(aHandle: THandle);
      procedure RemoveByIndex(Index:Integer);
      procedure Add(StickyWindow:TStickyElement);
      procedure Save(reg: TRegistry);
      procedure Load(reg: TRegistry);
      property Items[Index:Integer]:TStickyElement read RItems; default;
      property Count:Integer read RCount;
  End;

implementation
Uses
  SysUtils, Win2Tree;

function GetExprElement(Var Expr:String):String;
Begin
  If Pos(',', Expr)>0 Then Begin
    Result:=Copy(Expr, 1, Pos(',', Expr)-1);
    Delete(Expr, 1, Pos(',', Expr));
  End Else Begin
    Result:=Expr;
    Expr:='';
  End;
  Result:=Trim(Result);
End;

function ParseExpr(Expr: String; Title:String):Boolean;
Begin
  Expr:=UpperCase(Expr); Title:=UpperCase(Title);
  Result:=Expr<>'';
  While (Expr<>'')and(Result) Do
    Result:=Result and (Pos(GetExprElement(Expr), Title)>0);
End;

{******************************************************************************}
{*                                                                            *}
{*  TStickyElement                                                            *}
{*                                                                            *}
{******************************************************************************}
function TStickyElement.IsSticky(aHandle: THandle):Boolean; register;
begin
  Result := False;
  Case FMode of
    semHandle: Result := FHandle=aHandle;
    semClass : Result := FEXEName=UpperCase(GetWindowClass(aHandle));
    semEXPR  : Result := ParseExpr(FExpression, GetWindowTitle(aHandle));
  End;
end; {-- TStickyElement.IsSicky -----------------------------------------------}

constructor TStickyElement.Create(aHandle:THandle; Const aEXEname, aExpr: String; aMode: TStickyElementMode);
begin
  Inherited Create;

  FHandle     := aHandle;
  FEXEName    := UpperCase(aEXEName);
  FExpression := aExpr;
  FMode       := aMode;
end; {-- TStickyElement.Create ------------------------------------------------}

procedure TStickyElement.Save(reg: TRegistry; No:Integer);
Var
  Param : String;
begin
  reg.WriteInteger(Format('StickyWindows.%d.Mode', [No]), Integer(Mode));
  Case Mode of
    semClass: Param:=EXEName;
    semEXPR : Param:=Expression;
    Else      Param:='';
  End;
  reg.WriteString(Format('StickyWindows.%d.Param', [No]), Param);
end; {-- TStickyElement.Save --------------------------------------------------}

constructor TStickyElement.Load(reg: TRegistry; No:Integer);
Var
  Param : String;
begin
  Inherited Create;

  FHandle     := -1;
  FEXEName    := '';
  FExpression := '';
  FMode       := semEXPR;
  try
    FMode := TStickyElementMode(reg.ReadInteger(Format('StickyWindows.%d.Mode', [No])));
    Param := reg.ReadString(Format('StickyWindows.%d.Param', [No]));
    Case FMode of
      semClass: EXEName    := Param;
      semEXPR : Expression := Param;
    End;
  except end;
end; {-- TStickyElement.Load --------------------------------------------------}

{******************************************************************************}
{*                                                                            *}
{*  TStickyElement                                                            *}
{*                                                                            *}
{******************************************************************************}
function TStickyWindows.RItems(Index:Integer):TStickyElement; register;
begin
  If Index<Count Then Result:=FWindows[Index] Else Result:=nil;
end; {-- TStickyWindows.RItems ------------------------------------------------}

function TStickyWindows.RCount:Integer; register;
begin
  Result:=FWindows.Count;
end; {-- TStickyWindows.RCount ------------------------------------------------}

constructor TStickyWindows.Create;
begin
  Inherited Create;

  FWindows := TList.Create;
end; {-- TStickyWindows.Create ------------------------------------------------}

destructor TStickyWindows.Destroy;
Var
  I: Integer;
begin
  For I:=0 To Count-1 Do TStickyElement(Items[I]).Destroy;
  FWindows.Destroy;

  Inherited Destroy;
end; {-- TStickyWindows.Destroy -----------------------------------------------}

function TStickyWindows.IsSticky(aHandle: THandle):Boolean; register;
Var
  I: Integer;
Begin
  For I:=0 To Count-1 Do Begin
    Result:=TStickyElement(Items[I]).IsSticky(aHandle);
    If Result Then Exit;
  End;
  Result:=False;
End; {-- TStickyWindows.IsSicky -----------------------------------------------}

procedure TStickyWindows.Remove(aHandle: THandle);
Var
  I: Integer;
Begin
  For I:=0 To Count-1 Do
    If TStickyElement(Items[I]).IsSticky(aHandle) Then FWindows[I]:=nil;
  FWindows.Pack;
End; {-- TStickyWindows.Remove ------------------------------------------------}

procedure TStickyWindows.RemoveByIndex(Index:Integer);
Begin
  If Index<Count Then Begin
    TStickyElement(FWindows.Items[Index]).Destroy;
    FWindows.Items[Index]:=nil;
    FWindows.Pack;
  End;
End; {-- TStickyWindows.RemoveByIndex -----------------------------------------}

procedure TStickyWindows.Add(StickyWindow:TStickyElement);
Begin
  FWindows.Add(StickyWindow);
End; {-- TStickyWindows.Add ---------------------------------------------------}

procedure TStickyWindows.Save(reg: TRegistry);
Var
  I, J: Integer;
Begin
  J:=0; For I:=0 To Count-1 Do
    If Items[I].Mode<>semHandle Then Begin
      Items[I].Save(reg, J);
      Inc(J);
    End;
  reg.WriteInteger('StickyWindows.Count', J);
End; {-- TStickyWindows.Save --------------------------------------------------}

procedure TStickyWindows.Load(reg: TRegistry);
Var
  I  : Integer;
  SE : TStickyElement;
Begin
  try
    I:=reg.ReadInteger('StickyWindows.Count');
  except I:=0; End;
  For I:=0 To I-1 Do Begin
    SE := TStickyElement.Load(reg, I);
    If (SE.Expression='')and(SE.EXEName='') Then SE.Destroy Else Add(SE);
  End;
End; {-- TStickyWindows.Load --------------------------------------------------}

end.
