{*******************************************************}
{                                                       }
{         Delphi VCL Extensions (RX)                    }
{                                                       }
{         Copyright (c) 1995, 1996 AO ROSNO             }
{         Copyright (c) 1997 Master-Bank                }
{                                                       }
{*******************************************************}

unit Parsing;

interface

{$I RX.INC}

uses SysUtils;

type
  TParserFunc = (pfArcTan, pfCos, pfSin, pfTan, pfAbs, pfExp, pfLn, pfLog,
    pfSqrt, pfSqr, pfInt, pfFrac, pfTrunc, pfRound, pfArcSin, pfArcCos,
    pfSign, pfNot);

  TRxMathParser = class(TObject)
  private
    FCurPos: Cardinal;
    FParseText: string;
    function GetChar: Char;
    procedure NextChar;
    function GetNumber(var AValue: Extended): Boolean;
    function GetConst(var AValue: Extended): Boolean;
    function GetFunction(var AValue: TParserFunc): Boolean;
    function Term: Extended;
    function SubTerm: Extended;
    function Calculate: Extended;
  public
    function Exec(const AFormula: string): Extended;
  end;

  ERxParserError = class(Exception);

function GetFormulaValue(const Formula: string): Extended;

{$IFNDEF WIN32}
function Power(Base, Exponent: Extended): Extended;
{$ENDIF}

implementation

uses RXTConst;

const
  SpecialChars = [#0..' ', '+', '-', '/', '*', ')', '^'];

  FuncNames: array[TParserFunc] of PChar =
    ('ARCTAN', 'COS', 'SIN', 'TAN', 'ABS', 'EXP', 'LN', 'LOG',
    'SQRT', 'SQR', 'INT', 'FRAC', 'TRUNC', 'ROUND', 'ARCSIN', 'ARCCOS',
    'SIGN', 'NOT');

{ Parser errors }

procedure InvalidCondition(Str: Word);
begin
  raise ERxParserError.Create(LoadStr(Str));
end;

{ IntPower and Power functions are copied from Borland's MATH.PAS unit }

function IntPower(Base: Extended; Exponent: Integer): Extended;
{$IFDEF WIN32}
asm
        mov     ecx, eax
        cdq
        fld1                      { Result := 1 }
        xor     eax, edx
        sub     eax, edx          { eax := Abs(Exponent) }
        jz      @@3
        fld     Base
        jmp     @@2
@@1:    fmul    ST, ST            { X := Base * Base }
@@2:    shr     eax,1
        jnc     @@1
        fmul    ST(1),ST          { Result := Result * X }
        jnz     @@1
        fstp    st                { pop X from FPU stack }
        cmp     ecx, 0
        jge     @@3
        fld1
        fdivrp                    { Result := 1 / Result }
@@3:
        fwait
end;
{$ELSE}
var
  Y: Longint;
begin
  Y := Abs(Exponent);
  Result := 1.0;
  while Y > 0 do begin
    while not Odd(Y) do begin
      Y := Y shr 1;
      Base := Base * Base;
    end;
    Dec(Y);
    Result := Result * Base;
  end;
  if Exponent < 0 then Result := 1.0 / Result;
end;
{$ENDIF WIN32}

function Power(Base, Exponent: Extended): Extended;
begin
  if Exponent = 0.0 then Result := 1.0
  else if (Base = 0.0) and (Exponent > 0.0) then Result := 0.0
  else if (Frac(Exponent) = 0.0) and (Abs(Exponent) <= MaxInt) then
    Result := IntPower(Base, Trunc(Exponent))
  else Result := Exp(Exponent * Ln(Base))
end;

{ Parsing routines }

function GetFormulaValue(const Formula: string): Extended;
var
  MathParser: TRxMathParser;
begin
  MathParser := TRxMathParser.Create;
  try
    Result := MathParser.Exec(Formula);
  finally
    MathParser.Free;
  end;
end;

{ TRxMathParser }

function TRxMathParser.GetChar: Char;
begin
  Result := FParseText[FCurPos];
end;

procedure TRxMathParser.NextChar;
begin
  Inc(FCurPos);
end;

{$IFDEF WIN32}
  {$HINTS OFF}
{$ENDIF}
function TRxMathParser.GetNumber(var AValue: Extended): Boolean;
var
  C: Char;
  SavePos: Cardinal;
  Value: Extended;
  Code: Integer;
  TmpStr: String;
begin
  Result := False;
  C := GetChar;
  SavePos := FCurPos;
  if C in ['0'..'9', '.', '+', '-'] then begin
    TmpStr := '';
    if (C = '.') then TmpStr := '0' + C else TmpStr := C;
    NextChar;
    C := GetChar;
    if (Length(TmpStr) = 1) and (TmpStr[1] in ['+', '-']) and
      (C = '.') then TmpStr := TmpStr + '0';
    while C in ['0'..'9', '.', 'E'] do begin
      TmpStr := TmpStr + C;
      if (C = 'E') then begin
        if (Length(TmpStr) > 1) and (TmpStr[Length(TmpStr) - 1] = '.') then
          Insert('0', TmpStr, Length(TmpStr));
        NextChar;
        C := GetChar;
        if (C in ['+', '-']) then begin
          TmpStr := TmpStr + C;
          NextChar;
        end;
      end
      else NextChar;
      C := GetChar;
    end;
    if (TmpStr[Length(TmpStr)] = '.') and (Pos('E', TmpStr) = 0) then
      TmpStr := TmpStr + '0';
    Val(TmpStr, Value, Code);
    if Code <> 0 then FCurPos := SavePos
    else Result := True;
  end;
  if Result and not (FParseText[FCurPos] in SpecialChars) then begin
    FCurPos := SavePos;
    Result := False;
  end;
  if Result then AValue := StrToFloat(TmpStr)
  else AValue := 0;
end;
{$IFDEF WIN32}
  {$HINTS ON}
{$ENDIF}

function TRxMathParser.GetConst(var AValue: Extended): Boolean;
begin
  Result := False;
  case FParseText[FCurPos] of
    'E':
      if FParseText[FCurPos + 1] in SpecialChars then
      begin
        AValue := Exp(1);
        Inc(FCurPos);
        Result := True;
      end;
    'P':
      if (FParseText[FCurPos + 1] = 'I') and
        (FParseText[FCurPos + 2] in SpecialChars) then
      begin
        AValue := Pi;
        Inc(FCurPos, 2);
        Result := True;
      end;
  end
end;

function TRxMathParser.GetFunction(var AValue: TParserFunc): Boolean;
var
  I: TParserFunc;
  TmpStr: string;
begin
  Result := False;
  AValue := Low(TParserFunc);
  if FParseText[FCurPos] in ['A'..'Z'] then begin
    for I := Low(TParserFunc) to High(TParserFunc) do begin
      TmpStr := Copy(FParseText, FCurPos, StrLen(FuncNames[I]));
      if CompareStr(TmpStr, StrPas(FuncNames[I])) = 0 then begin
        AValue := I;
        if FParseText[FCurPos + Cardinal(Length(TmpStr))] = '(' then begin
          Result := True;
          Inc(FCurPos, Length(TmpStr));
          Break;
        end;
      end;
    end;
  end;
end;

function TRxMathParser.Term: Extended;
var
  Value: Extended;
  NoFunc: TParserFunc;
begin
  if FParseText[FCurPos] = '(' then begin
    Inc(FCurPos);
    Value := Calculate;
    if FParseText[FCurPos] <> ')' then InvalidCondition(SParseNotCramp);
    Inc(FCurPos);
  end
  else begin
    if not GetNumber(Value) then
      if not GetConst(Value) then
        if GetFunction(NoFunc) then begin
          Inc(FCurPos);
          Value := Calculate;
          try
            case NoFunc of
              pfArcTan: Value := ArcTan(Value);
              pfCos: Value := Cos(Value);
              pfSin: Value := Sin(Value);
              pfTan:
                if Cos(Value) = 0 then InvalidCondition(SParseDivideByZero)
                else Value := Sin(Value) / Cos(Value);
              pfAbs: Value := Abs(Value);
              pfExp: Value := Exp(Value);
              pfLn:
                if Value <= 0 then InvalidCondition(SParseLogError)
                else Value := Ln(Value);
              pfLog:
                if Value <= 0 then InvalidCondition(SParseLogError)
                else Value := Ln(Value) / Ln(10);
              pfSqrt:
                if Value < 0 then InvalidCondition(SParseSqrError)
                else Value := Sqrt(Value);
              pfSqr: Value := Sqr(Value);
              pfInt: Value := Round(Value);
              pfFrac: Value := Frac(Value);
              pfTrunc: Value := Trunc(Value);
              pfRound: Value := Round(Value);
              pfArcSin:
                if Value = 1 then Value := Pi / 2
                else Value := ArcTan(Value / Sqrt(1 - Sqr(Value)));
              pfArcCos:
                if Value = 1 then Value := 0
                else Value := Pi / 2 - ArcTan(Value / Sqrt(1 - Sqr(Value)));
              pfSign:
                if Value > 0 then Value := 1
                else if Value < 0 then Value := -1;
              pfNot: Value := not Trunc(Value);
            end;
          except
            on E: ERxParserError do raise
            else InvalidCondition(SParseInvalidFloatOperation);
          end;
          if FParseText[FCurPos] <> ')' then InvalidCondition(SParseNotCramp);
          Inc(FCurPos);
        end
        else InvalidCondition(SParseSyntaxError);
  end;
  Result := Value;
end;

function TRxMathParser.SubTerm: Extended;
var
  Value: Extended;
begin
  Value := Term;
  while FParseText[FCurPos] in ['*', '^', '/'] do begin
    Inc(FCurPos);
    if FParseText[FCurPos - 1] = '*' then
      Value := Value * Term
    else if FParseText[FCurPos - 1] = '^' then
      Value := Power(Value, Term)
    else if FParseText[FCurPos - 1] = '/' then
      try
        Value := Value / Term;
      except
        InvalidCondition(SParseDivideByZero);
      end;
  end;
  Result := Value;
end;

function TRxMathParser.Calculate: Extended;
var
  Value: Extended;
begin
  Value := SubTerm;
  while FParseText[FCurPos] in ['+','-'] do begin
    Inc(FCurPos);
    if FParseText[FCurPos - 1] = '+' then Value := Value + SubTerm
    else Value := Value - SubTerm;
  end;
  if not (FParseText[FCurPos] in [#0, ')', '>', '<', '=', ',']) then
    InvalidCondition(SParseSyntaxError);
  Result := Value;
end;

function TRxMathParser.Exec(const AFormula: string): Extended;
var
  I, J: Integer;
begin
  J := 0;
  Result := 0;
  FParseText := '';
  for I := 1 to Length(AFormula) do begin
    case AFormula[I] of
      '(': Inc(J);
      ')': Dec(J);
    end;
    if AFormula[I] > ' ' then FParseText := FParseText + UpCase(AFormula[I]);
  end;
  if J = 0 then begin
    FCurPos := 1;
    FParseText := FParseText + #0;
    Result := Calculate;
  end
  else InvalidCondition(SParseNotCramp);
end;

end.